<?php
class DSR_License_Handler {
    private $menu_registered = false; // Track if the menu is already registered

    public function __construct() {
        error_log('DSR_License_Handler instantiated');
        add_action('admin_menu', array($this, 'add_license_menu'));
    }

    public function add_license_menu() {
        if ($this->menu_registered) {
            error_log('Licensing & Activation submenu already registered, skipping');
            return;
        }

        error_log('Attempting to register Licensing & Activation submenu with permissions check');
        add_submenu_page(
            'xpertlync-dsr-form',
            __('Licensing & Activation', 'dsr-form'),
            __('Licensing & Activation', 'dsr-form'),
            'manage_options',
            'licensing-activation',
            'display_license_activation_page'
        );
        error_log('Licensing & Activation submenu registered successfully with permissions check');

        // Debug submenu registration
        global $submenu;
        error_log(print_r($submenu, true));

        $this->menu_registered = true; // Mark menu as registered
    }

    // Add AES-256 encryption/decryption methods to the class
    public static function encrypt_license_key($plain_text) {
        $key = hash('sha256', AUTH_KEY, true); // Use WordPress AUTH_KEY as base
        $iv = openssl_random_pseudo_bytes(16);
        $encrypted = openssl_encrypt($plain_text, 'aes-256-cbc', $key, OPENSSL_RAW_DATA, $iv);
        return base64_encode($iv . $encrypted);
    }
    public static function decrypt_license_key($encrypted_text) {
        $key = hash('sha256', AUTH_KEY, true);
        $data = base64_decode($encrypted_text);
        $iv = substr($data, 0, 16);
        $encrypted = substr($data, 16);
        return openssl_decrypt($encrypted, 'aes-256-cbc', $key, OPENSSL_RAW_DATA, $iv);
    }
}

// Update constants with the correct secret keys
define('SLM_SECRET_KEY_CREATION', '684dca3de36c70.52307811');
define('SLM_SECRET_KEY_VERIFICATION', '684dca3de36ce2.19470842');
define('SLM_LICENSE_SERVER_URL', 'http://xpertlync.com/wp/wp3-beta-testing');
define('SLM_ITEM_REFERENCE', 'xpertlync-dsr-form');

function display_license_activation_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.', 'dsr-form'));
    }

    // Check if a license key is submitted
    if (isset($_POST['activate_license'])) {
        // Always use the plain, user-entered license key for activation
        $license_key = sanitize_text_field($_POST['license_key']);

        // API query parameters (use plain key)
        $api_params = array(
            'slm_action' => 'slm_activate',
            'secret_key' => SLM_SECRET_KEY_VERIFICATION,
            'license_key' => $license_key, // PLAIN key only
            'registered_domain' => $_SERVER['SERVER_NAME'],
            'item_reference' => urlencode(SLM_ITEM_REFERENCE),
        );

        // Send query to the license manager server
        $query = esc_url_raw(add_query_arg($api_params, SLM_LICENSE_SERVER_URL));
        $response = wp_remote_get($query, array('timeout' => 20, 'sslverify' => false));

        // Check for error in the response
        if (is_wp_error($response)) {
            $status = 'Unexpected Error! The query returned with an error.';
        } else {
            $license_data = json_decode(wp_remote_retrieve_body($response));            if ($license_data && $license_data->result === 'success') {
                // Only encrypt and store after successful activation
                $encrypted_license_key = DSR_License_Handler::encrypt_license_key($license_key);
                update_option('xpertlync_dsr_license_key', $encrypted_license_key);
                update_option('xpertlync_dsr_license_status', 'active');
                error_log('License activated successfully. Status set to active.');
                wp_cache_delete('alloptions', 'options');
                wp_cache_flush(); // Ensure all caches are cleared
                $status = 'License key activated successfully.';
                // Force refresh the page to show updated status
                wp_redirect(admin_url('admin.php?page=licensing-activation&activated=true'));
                exit;
            } else {
                $status = 'License activation failed: ' . ($license_data->message ?? 'Unknown error.');
                error_log('License activation failed. Status not updated.');
            }
        }
    }

    // When displaying the license key in the form, always decrypt
    $stored_license_key = get_option('xpertlync_dsr_license_key');
    $license_key = $stored_license_key ? DSR_License_Handler::decrypt_license_key($stored_license_key) : '';    $license_status = get_option('xpertlync_dsr_license_status');
    $is_active = ($license_status === 'active' && !empty($license_key));
    $return_url = admin_url('admin.php?page=licensing-activation&license_payment=success');
    echo '<div class="wrap" style="max-width:700px;">';
    echo '<h1 style="margin-bottom:0.5em;">' . __('Licensing & Activation', 'dsr-form') . '</h1>';
    // Side-by-side block container
    echo '<div class="dsr-license-flex-row" style="display:flex;gap:24px;flex-wrap:wrap;align-items:flex-start;margin-bottom:24px;">';
    // Must-Have Plugin block
    echo '<div class="dsr-license-block" style="flex:1 1 260px;min-width:220px;max-width:320px;background:#fff;border-radius:10px;box-shadow:0 2px 8px #0001;padding:20px 24px;">'
        . '<h2 style="margin-top:0;color:#0d6efd;font-size:1.1em;">Only $10 per domain – A Must-Have Compliance Plugin for Every Business!</h2>'
        . '<div style="font-size:15px;line-height:1.7;margin-bottom:10px;">No matter your business size or industry, if you\'re collecting user data, compliance isn’t optional – it\'s mandatory. The DSR Request Form Plugin is your all-in-one solution to stay fully compliant with GDPR, CCPA, and global privacy laws – without the legal headaches.</div>'
        . '<ul style="margin:0 0 10px 18px;padding:0;font-size:15px;line-height:1.7;">'
        . '<li><strong>Instant Compliance</strong> – Meet key GDPR (Articles 12, 15, 17) and CCPA requirements effortlessly.</li>'
        . '<li><strong>Automated DSR Handling</strong> – Accept, manage, and respond to data subject requests in a secure, streamlined way.</li>'
        . '<li><strong>Avoid Legal Penalties</strong> – Shield your business from costly fines and legal action.</li>'
        . '<li><strong>Boost Customer Trust</strong> – Show your users that their privacy rights matter.</li>'
        . '<li><strong>Easy Setup, No Coding</strong> – Just install, activate, and you\'re ready in minutes.</li>'
        . '</ul>'
        . '<div style="font-size:15px;margin-bottom:8px;">For just $10 per domain, you get a professional-grade plugin that helps you protect your business and earn your customers\' confidence.</div>'        . '<div style="font-size:16px;color:#389e0d;margin-bottom:8px;">🛡️ Compliance is not optional. Make it simple, affordable, and foolproof with the DSR Request Form plugin.</div>'
        . '<a href="https://xpertlync.com/buy-dsr-license" target="_blank" style="display:inline-block;background:#0d6efd;color:#fff;font-weight:600;padding:10px 24px;border-radius:6px;text-decoration:none;margin-top:8px;">Buy Now – Protect your business today!</a>'
        . '</div>';
    // License status + form block (buttons below input, modern style)
    echo '<div class="dsr-license-block" style="flex:1 1 320px;min-width:260px;max-width:380px;background:#fff;border-radius:10px;box-shadow:0 2px 8px #0001;padding:20px 24px;">';    if ($is_active) {
        echo '<span style="display:inline-block;padding:6px 18px;background:#28a745;color:#fff;border-radius:20px;font-weight:600;font-size:15px;">' . __('Activated', 'dsr-form') . '</span>';
    } else {
        echo '<span style="display:inline-block;padding:6px 18px;background:#dc3545;color:#fff;border-radius:20px;font-weight:600;font-size:15px;">' . __('License Required', 'dsr-form') . '</span>';
    }
    echo '<form method="post" style="margin:18px 0 0 0;">';
    echo '<div style="margin-bottom:16px;">';
    echo '<label for="license_key" style="font-weight:600;font-size:15px;">' . __('License Key', 'dsr-form') . '</label><br>';
    echo '<input type="text" id="license_key" name="license_key" value="' . esc_attr($license_key) . '" style="width:100%;padding:8px 12px;font-size:15px;border-radius:5px;border:1px solid #ccc;margin-top:4px;" placeholder="Enter your license key" ' . ($is_active ? 'readonly style=\"background:#e9ffe9;\"' : '') . ' />';
    echo '</div>';    echo '<div style="display:flex;gap:10px;flex-direction:row;">';
    echo '<button type="submit" name="activate_license" class="dsr-modern-btn-primary" style="flex:1;font-size:15px;padding:10px 0;">' . ($is_active ? __('Activated', 'dsr-form') : __('Activate License', 'dsr-form')) . '</button>';
    if (!$is_active) {
        echo '<a href="https://www.xpertlync.com/data-subject-form/" target="_blank" class="dsr-modern-btn-secondary" style="flex:1;font-size:15px;padding:10px 0;text-decoration:none;text-align:center;">Buy Now</a>';
    }
    echo '</div>';
    echo '</form>';
    if (isset($status)) {
        echo '<p style="margin-top:18px;font-size:15px;">' . esc_html($status) . '</p>';
    }
    if (!$is_active) {
        echo '<div class="notice notice-warning" style="margin-top:24px;">';
        echo '<p style="font-size:16px;font-weight:bold;color:#b94a48;">' . __('No valid license is active. Please request or activate your license to use all features.', 'dsr-form') . '</p>';
        echo '</div>';
    }
    echo '</div>';
    // Modern button styles
    echo '<style>.dsr-modern-btn-primary{background:#0d6efd;color:#fff;border:none;border-radius:6px;transition:background 0.2s;box-shadow:0 2px 8px #0d6efd22;cursor:pointer;font-weight:600;}.dsr-modern-btn-primary:hover{background:#0b5ed7;}.dsr-modern-btn-secondary{background:#fff;color:#0d6efd;border:1.5px solid #0d6efd;border-radius:6px;transition:background 0.2s,color 0.2s;box-shadow:0 2px 8px #0d6efd11;cursor:pointer;font-weight:600;}.dsr-modern-btn-secondary:hover{background:#e3f0ff;color:#0b5ed7;}</style>';
    echo '</div>';
    echo '<div style="font-size:14px;color:#888;margin-top:12px;">' . __('Need help? Contact support at support@xpertlync.com', 'dsr-form') . '</div>';
    echo '</div>';
    // Modern UI block styles
    echo '<style>.dsr-license-block{background:#fff;border-radius:10px;box-shadow:0 2px 8px #0001;padding:20px 24px;margin-bottom:18px;} .dsr-license-status-active{display:inline-block;padding:6px 18px;background:#28a745;color:#fff;border-radius:20px;font-weight:600;font-size:15px;} .dsr-license-status-required{display:inline-block;padding:6px 18px;background:#dc3545;color:#fff;border-radius:20px;font-weight:600;font-size:15px;}</style>';
    // Razorpay payment/activation automation
    if (isset($_GET['license_payment']) && $_GET['license_payment'] === 'success' && !$is_active) {
        echo '<div class="notice notice-success" style="margin-bottom:18px;"><p style="font-size:15px;font-weight:bold;">' . __('Payment successful! Activating your license...', 'dsr-form') . '</p></div>';
        $admin_email = get_option('admin_email');
        // Step 1: Request a new license key from the license server
        $create_params = array(
            'slm_action' => 'slm_create_new',
            'secret_key' => SLM_SECRET_KEY_CREATION,
            'item_reference' => urlencode(SLM_ITEM_REFERENCE),
            'email' => $admin_email,
            'registered_domain' => $_SERVER['SERVER_NAME']
        );
        $create_query = esc_url_raw(add_query_arg($create_params, SLM_LICENSE_SERVER_URL));
        $create_response = wp_remote_get($create_query, array('timeout' => 20, 'sslverify' => false));
        $license_key = '';
        if (!is_wp_error($create_response)) {
            $create_data = json_decode(wp_remote_retrieve_body($create_response));
            if ($create_data && isset($create_data->license_key)) {
                $license_key = sanitize_text_field($create_data->license_key);
                // Step 2: Activate the new license key
                $activate_params = array(
                    'slm_action' => 'slm_activate',
                    'secret_key' => SLM_SECRET_KEY_VERIFICATION,
                    'license_key' => $license_key,
                    'registered_domain' => $_SERVER['SERVER_NAME'],
                    'item_reference' => urlencode(SLM_ITEM_REFERENCE),
                );
                $activate_query = esc_url_raw(add_query_arg($activate_params, SLM_LICENSE_SERVER_URL));
                $activate_response = wp_remote_get($activate_query, array('timeout' => 20, 'sslverify' => false));
                if (!is_wp_error($activate_response)) {
                    $activate_data = json_decode(wp_remote_retrieve_body($activate_response));
                    if ($activate_data && $activate_data->result === 'success') {
                        update_option('xpertlync_dsr_license_key', $license_key);
                        update_option('xpertlync_dsr_license_status', 'active');
                        wp_cache_delete('alloptions', 'options');
                        echo '<div class="notice notice-success" style="margin-bottom:18px;"><p>' . __('Your license has been automatically activated! All features are now enabled.', 'dsr-form') . '</p></div>';
                        $is_active = true;
                    } else {
                        echo '<div class="notice notice-error" style="margin-bottom:18px;"><p>' . __('License activation failed: ', 'dsr-form') . esc_html($activate_data->message ?? 'Unknown error.') . '</p></div>';
                    }
                } else {
                    echo '<div class="notice notice-error" style="margin-bottom:18px;"><p>' . __('Could not connect to license server for activation.', 'dsr-form') . '</p></div>';
                }
            } else {
                echo '<div class="notice notice-error" style="margin-bottom:18px;"><p>' . __('Could not retrieve license key from server.', 'dsr-form') . '</p></div>';
            }
        } else {
            echo '<div class="notice notice-error" style="margin-bottom:18px;"><p>' . __('Could not connect to license server for license creation.', 'dsr-form') . '</p></div>';
        }
    }
}

// Remove duplicate Licensing & Activation menu
add_action('admin_menu', function() {
    global $submenu;
    if (isset($submenu['xpertlync-dsr-form'])) {
        foreach ($submenu['xpertlync-dsr-form'] as $key => $menu_item) {
            if ($menu_item[2] === 'licensing-activation' && $key > 0) {
                unset($submenu['xpertlync-dsr-form'][$key]);
            }
        }
    }
});

// Static methods for activation, deactivation, and uninstall
class DSR_License_Lifecycle {
    public static function activate() {
        delete_option('xpertlync_dsr_license_key');
        update_option('xpertlync_dsr_license_status', 'inactive');
        error_log('License key and status reset to inactive on fresh install.');
    }
    public static function deactivate() {
        delete_option('xpertlync_dsr_license_key');
        update_option('xpertlync_dsr_license_status', 'inactive');
        error_log('License key and status removed on plugin deactivation.');
    }
    public static function uninstall() {
        delete_option('xpertlync_dsr_license_key');
        delete_option('xpertlync_dsr_license_status');
        error_log('License key and status removed on plugin uninstall.');
    }
}


