<?php
class DSR_Form_Handler {
    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('wp_ajax_dsr_submit_request', array($this, 'handle_submission'));
        add_action('wp_ajax_nopriv_dsr_submit_request', array($this, 'handle_submission'));
        add_action('wp_ajax_dsr_load_region_fields', array($this, 'load_region_fields'));
        add_action('wp_ajax_nopriv_dsr_load_region_fields', array($this, 'load_region_fields'));
        add_action('wp_ajax_dsr_check_tracking_status', array($this, 'ajax_check_tracking_status'));
        add_action('wp_ajax_nopriv_dsr_check_tracking_status', array($this, 'ajax_check_tracking_status'));
    }
    
    public function enqueue_assets() {
        wp_enqueue_style(
            'dsr-form-style',
            DSR_PLUGIN_URL . 'assets/css/form-style.css',
            array(),
            DSR_VERSION
        );
        
        wp_enqueue_script(
            'dsr-form-script',
            DSR_PLUGIN_URL . 'assets/js/form-script.js',
            array('jquery'),
            DSR_VERSION,
            true
        );
        
        wp_localize_script('dsr-form-script', 'dsr_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('dsr_request_nonce'),
            'tracking_text' => __('Tracking ID', 'dsr-form'),
            'submit_text' => __('Submit', 'dsr-form')
        ));
    }
    
    public function handle_submission() {
        check_ajax_referer('dsr_request_nonce', 'security');

        // Validate GDPR consent
        if (empty($_POST['gdpr_consent'])) {
            wp_send_json_error(__('You must consent to data processing (GDPR).', 'dsr-form'));
        }

        $captcha_type = get_option('dsr_captcha_type', 'simple');
        if ($captcha_type === 'recaptcha') {
            $recaptcha_secret = get_option('dsr_recaptcha_secret_key', '');
            $recaptcha_response = isset($_POST['g-recaptcha-response']) ? $_POST['g-recaptcha-response'] : '';
            if (empty($recaptcha_response) || empty($recaptcha_secret)) {
                wp_send_json_error(__('Captcha is required.', 'dsr-form'));
            }
            $verify = wp_remote_post('https://www.google.com/recaptcha/api/siteverify', array(
                'body' => array(
                    'secret' => $recaptcha_secret,
                    'response' => $recaptcha_response,
                    'remoteip' => $_SERVER['REMOTE_ADDR']
                )
            ));
            $response_body = wp_remote_retrieve_body($verify);
            $result = json_decode($response_body, true);
            if (empty($result['success'])) {
                wp_send_json_error(__('Captcha verification failed.', 'dsr-form'));
            }
        } else {
            // Validate simple captcha (session-based)
            session_start();
            $user_captcha = isset($_POST['captcha']) ? trim($_POST['captcha']) : '';
            $valid_captcha = isset($_SESSION['dsr_captcha']) ? $_SESSION['dsr_captcha'] : '';
            if (empty($user_captcha) || strtolower($user_captcha) !== strtolower($valid_captcha)) {
                wp_send_json_error(__('Captcha is incorrect.', 'dsr-form'));
            }
        }

        // Handle file upload
        $attachment_url = '';
        $attachment_path = '';
        $debug_upload = '';
        if (!empty($_FILES['attachment']['name'])) {
            $uploaded = media_handle_upload('attachment', 0);
            if (is_wp_error($uploaded)) {
                $debug_upload = 'File upload failed: ' . $uploaded->get_error_message();
                wp_send_json_error(__('File upload failed.', 'dsr-form') . ' ' . $debug_upload);
            } else {
                $attachment_url = wp_get_attachment_url($uploaded);
                $attachment_path = get_attached_file($uploaded);
                $debug_upload = 'File uploaded successfully: ' . $attachment_url . ' | Path: ' . $attachment_path;
            }
        } else {
            $debug_upload = 'No file uploaded.';
        }

        $data = array(
            'user_type'    => sanitize_text_field($_POST['user_type']),
            'user_identity' => isset($_POST['user_identity']) ? sanitize_text_field($_POST['user_identity']) : '',
            'first_name'   => ucwords(strtolower(sanitize_text_field($_POST['first_name']))),
            'last_name'    => ucwords(strtolower(sanitize_text_field($_POST['last_name']))),
            'email'        => sanitize_email($_POST['email']),
            'country'      => sanitize_text_field($_POST['country']),
            'request_types' => isset($_POST['request_types']) ? array_map('sanitize_text_field', (array)$_POST['request_types']) : array(),
            'request_type' => isset($_POST['request_type']) ? sanitize_text_field($_POST['request_type']) : '',
            'aspects'      => isset($_POST['aspects']) ? array_map('sanitize_text_field', (array)$_POST['aspects']) : array(),
            'details'      => sanitize_textarea_field($_POST['details']),
            'attachment'   => $attachment_url,
            'attachment_path' => $attachment_path,
            'ip_address'   => $this->get_client_ip(),
            'gdpr_consent' => true
        );

        $db_handler = new DSR_DB_Handler();
        $tracking_id = $db_handler->save_request($data);

        $email_handler = new DSR_Email_Handler();
        $email_handler->send_confirmation($tracking_id);
        $email_handler->send_dpo_notification($tracking_id);

        wp_send_json_success(array(
            'message' => __('Your request has been submitted successfully!', 'dsr-form'),
            'tracking_id' => $tracking_id
        ));

        // Add debug info to response for troubleshooting
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[DSR] Attachment debug: ' . $debug_upload);
        }
    }
    
    public function load_region_fields() {
        check_ajax_referer('dsr_request_nonce', 'security');
        $region = isset($_POST['region']) ? sanitize_text_field($_POST['region']) : 'eu';
        ob_start();
        switch ($region) {
            case 'california':
                include DSR_PLUGIN_DIR . 'templates/forms/california-form.php';
                break;
            case 'other':
                include DSR_PLUGIN_DIR . 'templates/forms/other-form.php';
                break;
            case 'eu':
            default:
                include DSR_PLUGIN_DIR . 'templates/forms/eu-form.php';
                break;
        }
        $html = ob_get_clean();
        wp_send_json_success(['html' => $html]);
    }
    
    public function ajax_check_tracking_status() {
        $tracking_id = isset($_POST['tracking_id']) ? sanitize_text_field($_POST['tracking_id']) : '';
        if (empty($tracking_id)) {
            wp_send_json_error(__('Tracking ID is required.', 'dsr-form'));
        }
        global $wpdb;
        $table_name = $wpdb->prefix . 'dsr_requests';
        $request = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE tracking_id = %s", $tracking_id), ARRAY_A);
        if (!$request) {
            wp_send_json_error('Not found');
        }
        $data = maybe_unserialize($request['request_data']);
        $name = trim(($data['first_name'] ?? '') . ' ' . ($data['last_name'] ?? ''));
        $status = ($request['status'] === 'pending') ? __('Filed', 'dsr-form') : ucwords($request['status']);
        $filed_date = date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($request['created_at']));
        wp_send_json_success([
            'name' => $name,
            'status' => $status,
            'filed_date' => $filed_date
        ]);
    }
    
    private function get_client_ip() {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            return sanitize_text_field($_SERVER['HTTP_CLIENT_IP']);
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            return sanitize_text_field($_SERVER['HTTP_X_FORWARDED_FOR']);
        } else {
            return sanitize_text_field($_SERVER['REMOTE_ADDR']);
        }
    }
}